
const { Model, DataTypes } = require('sequelize');
const bcrypt = require('bcrypt');

class User extends Model {
  static init(sequelize) {
    return super.init({
      user_id: { type: DataTypes.BIGINT.UNSIGNED, autoIncrement: true, primaryKey: true },
      tenant_id: { type: DataTypes.UUID, allowNull: false },
      name: { type: DataTypes.STRING, allowNull: false },
      email: { type: DataTypes.STRING, allowNull: false },
      password_hash: { type: DataTypes.STRING },
      role: { 
        type: DataTypes.ENUM('musicfan','artist','label','editor','supervisor','manager','admin'),
        defaultValue: 'musicfan'
      },
      full_name: { type: DataTypes.STRING },
      address: { type: DataTypes.STRING },
      gov_id_url: { type: DataTypes.STRING },
      signature_url: { type: DataTypes.STRING },
      verification_status: {
        type: DataTypes.ENUM('unverified','pending','verified','rejected'),
        defaultValue: 'unverified'
      }
    }, { sequelize, tableName: 'users' });
  }

  static associate() {
    const { Role } = require('./index');
    this.belongsToMany(Role, { through: 'user_roles', foreignKey: 'user_id' });
  }

  async setPassword(password) {
    const saltRounds = parseInt(process.env.BCRYPT_SALT_ROUNDS || '10', 10);
    this.password_hash = await bcrypt.hash(password, saltRounds);
  }

  validPassword(password) {
    return bcrypt.compare(password, this.password_hash);
  }

  static async promoteToArtist(userId) {
    const user = await this.findByPk(userId);
    if (!user) throw new Error('User not found');
    if (user.verification_status !== 'verified') throw new Error('User not verified');
    user.role = 'artist';
    await user.save();
    return user;
  }
}

module.exports = User;
